import os
import uuid
import shutil
import json
import re
import fitz  # PyMuPDF
from datetime import datetime, timedelta, UTC
from typing import Optional, List, Dict, Any
from fastapi import APIRouter, Depends, HTTPException, status, Body, Request, Header, UploadFile, File
from fastapi.security import OAuth2PasswordBearer, OAuth2PasswordRequestForm, HTTPBasic, HTTPBasicCredentials
from fastapi.responses import JSONResponse
from pydantic import BaseModel, Field, EmailStr
from pymongo import MongoClient, DESCENDING
from jose import JWTError, jwt
from passlib.context import CryptContext
from bson import ObjectId
import secrets
from database import client, db, conversations_collection, admin_collection, settings_collection
from models import ChatbotSettings

# Create indexes for better performance
admin_collection.create_index("email", unique=True)

# Password and JWT utilities
pwd_context = CryptContext(
    schemes=["bcrypt"],
    deprecated="auto",
    bcrypt__rounds=12,  # Explicitly set rounds
    bcrypt__ident="2b"  # Use the stable version identifier
)
oauth2_scheme = OAuth2PasswordBearer(tokenUrl="/api/admin/login")

# JWT Settings
SECRET_KEY = os.environ.get("JWT_SECRET_KEY", "insecurelocalsecretkey12345")  # Use env var in production
ALGORITHM = "HS256"
ACCESS_TOKEN_EXPIRE_MINUTES = 30

# Developer Token for special endpoints (ideally, set this in environment variables)
DEVELOPER_TOKEN = os.environ.get("DEVELOPER_TOKEN", "dev_token_X392nZ7qKpM4T5A6")  # Change this!

# Data models
class AdminBase(BaseModel):
    email: EmailStr
    full_name: str

class AdminCreate(AdminBase):
    password: str

class DevAdminCreate(AdminCreate):
    role: str = "admin"  # Default role is admin

class Admin(AdminBase):
    id: str
    created_at: datetime
    is_active: bool = True

    class Config:
        from_attributes = True

class Token(BaseModel):
    access_token: str
    token_type: str

class TokenData(BaseModel):
    email: Optional[str] = None

class ChatSessionInfo(BaseModel):
    session_id: str
    created_at: datetime
    message_count: int

class ChatMessage(BaseModel):
    timestamp: datetime
    question: str
    answer: str

class SessionDetail(BaseModel):
    session_id: str
    created_at: datetime
    messages: List[ChatMessage]

class Stats(BaseModel):
    total_sessions: int
    total_messages: int
    active_today: int

# Password and auth utilities
def get_password_hash(password):
    return pwd_context.hash(password)

def verify_password(plain_password, hashed_password):
    return pwd_context.verify(plain_password, hashed_password)

def get_admin_by_email(email):
    admin = admin_collection.find_one({"email": email})
    if admin:
        admin["id"] = str(admin.pop("_id"))
        return admin
    return None

def authenticate_admin(email: str, password: str):
    admin = get_admin_by_email(email)
    if not admin:
        return False
    if not verify_password(password, admin["password"]):
        return False
    return admin

def create_access_token(data: dict, expires_delta: Optional[timedelta] = None):
    to_encode = data.copy()
    if expires_delta:
        expire = datetime.now(UTC) + expires_delta
    else:
        expire = datetime.now(UTC) + timedelta(minutes=15)
    to_encode.update({"exp": expire})
    encoded_jwt = jwt.encode(to_encode, SECRET_KEY, algorithm=ALGORITHM)
    return encoded_jwt

def verify_developer_token(token: str = Header(..., alias="X-Developer-Token")):
    if token != DEVELOPER_TOKEN:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Invalid developer token"
        )
    return True

async def get_current_admin(token: str = Depends(oauth2_scheme)):
    credentials_exception = HTTPException(
        status_code=status.HTTP_401_UNAUTHORIZED,
        detail="Could not validate credentials",
        headers={"WWW-Authenticate": "Bearer"},
    )
    try:
        payload = jwt.decode(token, SECRET_KEY, algorithms=[ALGORITHM])
        email: str = payload.get("sub")
        if email is None:
            raise credentials_exception
        token_data = TokenData(email=email)
    except JWTError:
        raise credentials_exception
    admin = get_admin_by_email(email=token_data.email)
    if admin is None:
        raise credentials_exception
    if not admin.get("is_active", True):
        raise HTTPException(status_code=400, detail="Inactive admin account")
    return admin

# Create router
admin_router = APIRouter(
    prefix="/api/admin",
    tags=["admin"],
)

# Ensure default admin account if none exists
def create_default_admin():
    # Delete existing default admin if it exists
    admin_collection.delete_one({"email": "admin@dlc.com"})
    
    # Create new default admin account
    admin = {
        "email": "admin@dlc.com",
        "full_name": "Default Admin",
        "password": get_password_hash("admin123"),  # This will now use bcrypt properly
        "created_at": datetime.now(UTC),
        "is_active": True,
        "role": "admin"
    }
    admin_collection.insert_one(admin)
    print("Created default admin account with proper password hashing!")

# Call on startup
create_default_admin()

# Developer API endpoint for creating admins
@admin_router.post("/dev/create-admin", response_model=Admin)
async def dev_create_admin(
    admin_data: DevAdminCreate,
    is_authenticated: bool = Depends(verify_developer_token)
):
    """
    Special endpoint for developers to create admin accounts.
    This is protected by a developer token and should not be used in production.
    """
    # Check if email already exists
    existing_admin = admin_collection.find_one({"email": admin_data.email})
    if existing_admin:
        raise HTTPException(status_code=400, detail="Email already registered")
    
    new_admin = {
        "email": admin_data.email,
        "full_name": admin_data.full_name,
        "password": get_password_hash(admin_data.password),
        "created_at": datetime.now(UTC),
        "is_active": True,
        "role": admin_data.role
    }
    
    result = admin_collection.insert_one(new_admin)
    new_admin["id"] = str(result.inserted_id)
    new_admin.pop("_id", None)
    new_admin.pop("password", None)
    
    return new_admin

# Auth endpoints
@admin_router.post("/login", response_model=Token)
async def login(login_data: OAuth2PasswordRequestForm = Depends()):
    admin = authenticate_admin(login_data.username, login_data.password)
    if not admin:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Incorrect email or password",
            headers={"WWW-Authenticate": "Bearer"},
        )
    access_token_expires = timedelta(minutes=ACCESS_TOKEN_EXPIRE_MINUTES)
    access_token = create_access_token(
        data={"sub": admin["email"]}, expires_delta=access_token_expires
    )
    return {"access_token": access_token, "token_type": "bearer"}

@admin_router.get("/me", response_model=Admin)
async def read_admin_me(token: str = Depends(oauth2_scheme)):
    try:
        payload = jwt.decode(token, SECRET_KEY, algorithms=[ALGORITHM])
        email: str = payload.get("sub")
        if email is None:
            raise HTTPException(status_code=401, detail="Invalid authentication credentials")
        token_data = TokenData(email=email)
    except JWTError:
        raise HTTPException(status_code=401, detail="Invalid authentication credentials")
    
    admin = get_admin_by_email(email=token_data.email)
    if admin is None:
        raise HTTPException(status_code=404, detail="Admin not found")
    return admin

@admin_router.post("/change-password")
async def change_password(
    current_password: str = Body(...),
    new_password: str = Body(...),
    current_admin: dict = Depends(get_current_admin)
):
    # Verify current password
    if not verify_password(current_password, current_admin["password"]):
        raise HTTPException(status_code=400, detail="Incorrect current password")
    
    # Update password
    hashed_password = get_password_hash(new_password)
    admin_collection.update_one(
        {"email": current_admin["email"]},
        {"$set": {"password": hashed_password}}
    )
    
    return {"message": "Password updated successfully"}

# Admin management endpoints
@admin_router.post("/create", response_model=Admin)
async def create_admin(
    admin_data: AdminCreate,
    current_admin: dict = Depends(get_current_admin)
):
    # Only existing admins can create new admins
    existing_admin = admin_collection.find_one({"email": admin_data.email})
    if existing_admin:
        raise HTTPException(status_code=400, detail="Email already registered")
    
    new_admin = {
        "email": admin_data.email,
        "full_name": admin_data.full_name,
        "password": get_password_hash(admin_data.password),
        "created_at": datetime.now(UTC),
        "is_active": True,
        "role": "admin"
    }
    
    result = admin_collection.insert_one(new_admin)
    new_admin["id"] = str(result.inserted_id)
    new_admin.pop("_id", None)
    new_admin.pop("password", None)
    
    return new_admin

@admin_router.get("/list", response_model=List[Admin])
async def list_admins(current_admin: dict = Depends(get_current_admin)):
    admins = []
    for admin in admin_collection.find():
        admin["id"] = str(admin.pop("_id"))
        admin.pop("password", None)  # Don't expose password hashes
        admins.append(admin)
    return admins

# Chat data access endpoints
@admin_router.get("/sessions", response_model=List[ChatSessionInfo])
async def get_sessions(
    limit: int = 20,
    skip: int = 0,
    current_admin: dict = Depends(get_current_admin)
):
    # Aggregate to get session info with message counts
    pipeline = [
        {"$group": {
            "_id": "$session_id",
            "created_at": {"$min": "$timestamp"},
            "message_count": {"$sum": 1}
        }},
        {"$sort": {"created_at": -1}},
        {"$skip": skip},
        {"$limit": limit},
        {"$project": {
            "_id": 0,
            "session_id": "$_id",
            "created_at": 1,
            "message_count": 1
        }}
    ]
    
    sessions = list(conversations_collection.aggregate(pipeline))
    return sessions

@admin_router.get("/sessions/{session_id}", response_model=SessionDetail)
async def get_session_detail(
    session_id: str,
    current_admin: dict = Depends(get_current_admin)
):
    # Get all messages for a specific session
    messages = list(conversations_collection.find(
        {"session_id": session_id},
        {"_id": 0, "context": 0}  # Exclude _id and context fields
    ).sort("timestamp", 1))
    
    if not messages:
        raise HTTPException(status_code=404, detail="Session not found")
    
    # Format the response
    session_detail = {
        "session_id": session_id,
        "created_at": messages[0]["timestamp"],
        "messages": messages
    }
    
    return session_detail

@admin_router.delete("/sessions/{session_id}")
async def delete_session(
    session_id: str,
    current_admin: dict = Depends(get_current_admin)
):
    result = conversations_collection.delete_many({"session_id": session_id})
    
    if result.deleted_count == 0:
        raise HTTPException(status_code=404, detail="Session not found")
    
    return {"message": f"Deleted {result.deleted_count} messages from session {session_id}"}

@admin_router.get("/stats", response_model=Stats)
async def get_stats(current_admin: dict = Depends(get_current_admin)):
    # Get basic stats about chat usage
    total_sessions = len(conversations_collection.distinct("session_id"))
    total_messages = conversations_collection.count_documents({})
    
    # Active sessions today
    today_start = datetime.now(UTC).replace(hour=0, minute=0, second=0, microsecond=0)
    active_today = len(conversations_collection.distinct(
        "session_id", 
        {"timestamp": {"$gte": today_start}}
    ))
    
    return {
        "total_sessions": total_sessions,
        "total_messages": total_messages,
        "active_today": active_today
    }

# Admin routes
@admin_router.get("/")
async def admin_home(admin: str = Depends(get_current_admin)):
    return {"message": "Admin access granted"}

@admin_router.get("/stats")
async def get_stats(admin: str = Depends(get_current_admin)):
    # Example stats - replace with actual metrics from your database
    return {
        "total_conversations": 0,
        "total_users": 0,
        "active_sessions": 0
    }

# Settings endpoints
@admin_router.get("/settings", response_model=ChatbotSettings)
async def get_settings(current_admin: dict = Depends(get_current_admin)):
    """Get current chatbot settings"""
    settings = settings_collection.find_one({})
    if not settings:
        # Return default settings if none exist
        settings = ChatbotSettings().dict()
        settings_collection.insert_one(settings)
    
    settings.pop("_id", None)  # Remove MongoDB ID
    return settings

@admin_router.post("/settings")
async def update_settings(
    settings: ChatbotSettings,
    current_admin: dict = Depends(get_current_admin)
):
    """Update chatbot settings"""
    try:
        settings_dict = settings.dict()
        settings_collection.replace_one({}, settings_dict, upsert=True)
        return {"message": "Settings updated successfully"}
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Error updating settings: {str(e)}")

@admin_router.post("/settings/reset")
async def reset_settings(current_admin: dict = Depends(get_current_admin)):
    """Reset settings to defaults"""
    try:
        default_settings = ChatbotSettings().dict()
        settings_collection.replace_one({}, default_settings, upsert=True)
        return {"message": "Settings reset to defaults"}
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Error resetting settings: {str(e)}")

# PDF Upload endpoints
@admin_router.post("/upload-pdfs")
async def upload_pdfs(
    pdfs: List[UploadFile] = File(...),
    current_admin: dict = Depends(get_current_admin)
):
    """Upload and process PDF files"""
    try:
        uploaded_count = 0
        processed_count = 0
        documents_dir = "documents"
        
        # Ensure documents directory exists
        os.makedirs(documents_dir, exist_ok=True)
        
        results = []
        
        for pdf in pdfs:
            if not pdf.filename.lower().endswith('.pdf'):
                continue
                
            # Save PDF to documents directory
            file_path = os.path.join(documents_dir, pdf.filename)
            with open(file_path, "wb") as buffer:
                shutil.copyfileobj(pdf.file, buffer)
            
            uploaded_count += 1
            
            # Process PDF to Q&A format
            success, message = process_pdf_to_qa(file_path, pdf.filename)
            if success:
                processed_count += 1
                results.append(f"✓ {pdf.filename}: {message}")
            else:
                results.append(f"✗ {pdf.filename}: {message}")
        
        return {
            "success": True,
            "uploaded_count": uploaded_count,
            "processed_count": processed_count,
            "message": f"Successfully uploaded {uploaded_count} PDF(s) and processed {processed_count} to Q&A format",
            "results": results
        }
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Error uploading PDFs: {str(e)}")

@admin_router.get("/uploaded-pdfs")
async def get_uploaded_pdfs(current_admin: dict = Depends(get_current_admin)):
    """Get list of uploaded PDF files and their corresponding Q&A files"""
    try:
        documents_dir = "documents"
        pdfs = []
        qa_files = []
        
        if os.path.exists(documents_dir):
            for filename in os.listdir(documents_dir):
                file_path = os.path.join(documents_dir, filename)
                if filename.lower().endswith('.pdf'):
                    file_size = os.path.getsize(file_path)
                    pdfs.append({
                        "filename": filename,
                        "size": file_size
                    })
                elif filename.lower().endswith('_qa.json'):
                    qa_files.append(filename)
        
        return {"pdfs": pdfs, "qa_files": qa_files}
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Error getting uploaded PDFs: {str(e)}")

@admin_router.delete("/delete-pdf/{filename}")
async def delete_pdf(
    filename: str,
    current_admin: dict = Depends(get_current_admin)
):
    """Delete a specific PDF file"""
    try:
        documents_dir = "documents"
        file_path = os.path.join(documents_dir, filename)
        
        if not os.path.exists(file_path):
            raise HTTPException(status_code=404, detail="PDF file not found")
        
        if not filename.lower().endswith('.pdf'):
            raise HTTPException(status_code=400, detail="Invalid file type")
        
        # Remove PDF file
        os.remove(file_path)
        
        # Also remove corresponding Q&A JSON file if it exists
        base_name = os.path.splitext(filename)[0]
        json_filename = f"{base_name}_qa.json"
        json_path = os.path.join(documents_dir, json_filename)
        if os.path.exists(json_path):
            os.remove(json_path)
        
        # Also remove corresponding paragraph chunks JSON file if it exists
        chunks_filename = f"{base_name}_chunks.json"
        chunks_path = os.path.join(documents_dir, chunks_filename)
        if os.path.exists(chunks_path):
            os.remove(chunks_path)
        
        return {
            "success": True,
            "message": f"PDF {filename} and associated Q&A file deleted successfully"
        }
    except HTTPException:
        raise
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Error deleting PDF: {str(e)}")

@admin_router.post("/rebuild-vectorstore")
async def rebuild_vectorstore(current_admin: dict = Depends(get_current_admin)):
    """Rebuild the vector store with all Q&A files"""
    try:
        # Set a flag to indicate vector store needs rebuilding
        # The main API will check this flag and rebuild when needed
        rebuild_flag_file = "vectorstore_rebuild_flag.txt"
        with open(rebuild_flag_file, "w") as f:
            f.write(str(datetime.now().isoformat()))
        
        return {
            "success": True,
            "message": "Vector store rebuild flag set. The chatbot will rebuild the vector store on the next request."
        }
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Error setting rebuild flag: {str(e)}")

# PDF Processing Functions
def extract_text_from_pdf(pdf_path):
    """Extract text from PDF file"""
    try:
        doc = fitz.open(pdf_path)
        text = ""
        for page in doc:
            text += page.get_text()
        doc.close()
        return text
    except Exception as e:
        print(f"Error extracting text from {pdf_path}: {str(e)}")
        return None

def create_qa_pairs_from_text(text, filename):
    """Create Q&A pairs from extracted text"""
    try:
        qa_pairs = []
        
        if not text:
            return qa_pairs
        
        # Clean filename for intent naming
        base_name = os.path.splitext(filename)[0].lower()
        base_name = re.sub(r'[^a-z0-9]', '_', base_name)
        
        # Split text into paragraphs
        paragraphs = [p.strip() for p in text.split('\n\n') if p.strip() and len(p.strip()) > 50]
        
        # Create Q&A pairs from paragraphs
        for i, paragraph in enumerate(paragraphs[:20]):  # Limit to first 20 paragraphs
            if len(paragraph) < 100:  # Skip very short paragraphs
                continue
                
            # Generate questions based on content
            questions = []
            
            # Extract key terms for questions
            sentences = paragraph.split('. ')
            if sentences:
                first_sentence = sentences[0]
                if len(first_sentence) > 20:
                    # Create question from first sentence
                    question = first_sentence.replace(' is ', ' is ').replace(' are ', ' are ')
                    if not question.endswith('?'):
                        question = f"What is {question.lower()}?"
                    questions.append(question)
            
            # Add generic questions
            questions.extend([
                f"What does the document say about {base_name.replace('_', ' ')}?",
                f"Tell me about {base_name.replace('_', ' ')}",
                f"What information is available about {base_name.replace('_', ' ')}?"
            ])
            
            # Create Q&A pair
            qa_pairs.append({
                "intent": f"{base_name}_section_{i+1}",
                "questions": questions,
                "answer": paragraph
            })
        
        # Add general document questions
        qa_pairs.append({
            "intent": f"{base_name}_overview",
            "questions": [
                f"What is this document about?",
                f"Tell me about {base_name.replace('_', ' ')}",
                f"What does the {base_name.replace('_', ' ')} document contain?",
                f"Give me an overview of {base_name.replace('_', ' ')}"
            ],
            "answer": f"This document contains information about {base_name.replace('_', ' ')}. It includes various sections covering different aspects of the topic."
        })
        
        return qa_pairs
    except Exception as e:
        print(f"Error creating Q&A pairs: {str(e)}")
        return []

def split_text_into_paragraphs(text):
    # Split text into paragraphs (by double newlines)
    return [p.strip() for p in text.split('\n\n') if p.strip()]

def process_pdf_to_qa(pdf_path, filename):
    """Process PDF file and create Q&A JSON and paragraph chunks JSON"""
    try:
        # Extract text from PDF
        text = extract_text_from_pdf(pdf_path)
        if not text:
            return False, "Failed to extract text from PDF"
        
        # Create Q&A pairs
        qa_pairs = create_qa_pairs_from_text(text, filename)
        if not qa_pairs:
            return False, "No Q&A pairs could be created from the PDF"
        
        # Save Q&A pairs to JSON file
        base_name = os.path.splitext(filename)[0]
        json_filename = f"{base_name}_qa.json"
        json_path = os.path.join("documents", json_filename)
        with open(json_path, "w", encoding="utf-8") as f:
            json.dump(qa_pairs, f, indent=4, ensure_ascii=False)

        # Save paragraph chunks to JSON file
        paragraphs = split_text_into_paragraphs(text)
        chunk_data = [{"filename": filename, "chunk": p, "index": i} for i, p in enumerate(paragraphs) if len(p) > 50]
        chunks_filename = f"{base_name}_chunks.json"
        chunks_path = os.path.join("documents", chunks_filename)
        with open(chunks_path, "w", encoding="utf-8") as f:
            json.dump(chunk_data, f, indent=4, ensure_ascii=False)

        return True, f"Successfully created {len(qa_pairs)} Q&A pairs and {len(chunk_data)} paragraph chunks"
    except Exception as e:
        return False, f"Error processing PDF: {str(e)}"

def process_all_existing_pdfs_to_chunks():
    """Process all existing PDFs in the documents directory to generate paragraph chunks JSON files if missing."""
    documents_dir = "documents"
    for filename in os.listdir(documents_dir):
        if filename.lower().endswith('.pdf'):
            base_name = os.path.splitext(filename)[0]
            chunks_filename = f"{base_name}_chunks.json"
            chunks_path = os.path.join(documents_dir, chunks_filename)
            pdf_path = os.path.join(documents_dir, filename)
            if not os.path.exists(chunks_path):
                print(f"Processing {filename} to generate paragraph chunks...")
                process_pdf_to_qa(pdf_path, filename)

# Export the router to be included in the main app
# In api.py, add: from admin import admin_router; app.include_router(admin_router) 