import os
import glob
import hashlib
import json
import uuid
from datetime import datetime, UTC
from fastapi import FastAPI, HTTPException, Request, Depends
from fastapi.middleware.cors import CORSMiddleware
from fastapi.staticfiles import StaticFiles
from fastapi.templating import Jinja2Templates
from fastapi.responses import HTMLResponse, RedirectResponse
from pydantic import BaseModel
from typing import List, Optional, Dict, Any
from admin import admin_router, SECRET_KEY, ALGORITHM, get_admin_by_email
from jose import JWTError, jwt
import time
import threading
from models import ChatbotSettings, ModelProvider
from intention import find_best_match, model, qa_data
import difflib
from sentence_transformers import SentenceTransformer
from database import (
    insert_conversation,
    get_session_list,
    get_session_history,
    get_settings,
    update_settings
)

# Create FastAPI app
app = FastAPI(
    title="DLC Chatbot API",
    description="API for ABU Distance Learning Centre Chatbot",
    version="1.0.0"
)

# Include admin routes
app.include_router(admin_router)

# Set up templates and static files
templates = Jinja2Templates(directory="templates")
os.makedirs("static", exist_ok=True)
app.mount("/static", StaticFiles(directory="static"), name="static")

# Add CORS middleware
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],  # Update with specific origins in production
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# Simple in-memory cache for responses
RESPONSE_CACHE = {}
CACHE_MAX_SIZE = 100  # Maximum number of items to keep in cache

# Data models for API
class QuestionRequest(BaseModel):
    question: str
    session_id: Optional[str] = None

class ChatResponse(BaseModel):
    answer: str
    session_id: str

class SessionResponse(BaseModel):
    session_id: str
    created_at: datetime

class SessionListResponse(BaseModel):
    sessions: List[SessionResponse]

class Question(BaseModel):
    query: str

class Answer(BaseModel):
    answer: str
    intent: Optional[str] = None
    confidence: Optional[float] = None

# Initialize default settings if none exist
default_settings = ChatbotSettings(
    provider=ModelProvider.INTENTION,
    model_name="paraphrase-MiniLM-L6-v2",
    similarity_threshold=0.6,
    debug_mode=False,
    log_conversations=True,
    enable_response_cache=True,
    cache_max_size=1000
)

if not get_settings():
    update_settings(default_settings.dict())

# Session management
def create_session():
    return str(uuid.uuid4())

# Chat endpoints
@app.post("/api/chat", response_model=ChatResponse)
async def chat(request: QuestionRequest):
    try:
        # Create session if not provided
        session_id = request.session_id or str(uuid.uuid4())
        
        # Get current settings
        settings = get_settings()
        if not settings:
            settings = default_settings.dict()
            update_settings(settings)
        
        # Check cache if enabled
        if settings.get('enable_response_cache', True):
        cache_key = f"{request.question.strip().lower()}"
        if cache_key in RESPONSE_CACHE:
            cached_response = RESPONSE_CACHE[cache_key]
            # Store in conversation history
                insert_conversation(
                    session_id=session_id,
                    question=request.question,
                    answer=cached_response,
                    source="cache"
                )
            return ChatResponse(answer=cached_response, session_id=session_id)
        
        # Get response using intention-based matching
        match = find_best_match(request.question, qa_data)
        answer = match.get('answer') if match else settings.get('default_response', "I'm sorry, I couldn't find a good match for your question. Could you please rephrase it?")
        
        # Store in cache if enabled and question isn't too long
        if settings.get('enable_response_cache', True) and len(request.question) < 200:
            cache_key = f"{request.question.strip().lower()}"
            RESPONSE_CACHE[cache_key] = answer
            # Limit cache size
            cache_max_size = settings.get('cache_max_size', 1000)
            if len(RESPONSE_CACHE) > cache_max_size:
                RESPONSE_CACHE.pop(next(iter(RESPONSE_CACHE)))
        
        # Store in conversation history if enabled
        if settings.get('log_conversations', True):
            insert_conversation(
                session_id=session_id,
                question=request.question,
                answer=answer,
                source="intention"
            )
        
        return ChatResponse(answer=answer, session_id=session_id)
        
    except Exception as e:
        print(f"Error in chat endpoint: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))

@app.get("/api/sessions", response_model=SessionListResponse)
async def list_sessions():
    try:
        sessions = get_session_list()
        return {"sessions": sessions}
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Error retrieving sessions: {str(e)}")

@app.get("/api/sessions/{session_id}/history")
async def get_chat_history(session_id: str):
    try:
        history = get_session_history(session_id)
        
        # Convert timestamps to ISO format strings
        for item in history:
            item["timestamp"] = item["timestamp"].isoformat()
            
        return {"session_id": session_id, "history": history}
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Error retrieving session history: {str(e)}")

# Web UI Routes
@app.get("/", response_class=HTMLResponse)
async def chat_ui(request: Request):
    return templates.TemplateResponse("index.html", {"request": request})

# Admin UI Routes
@app.get("/admin/login", response_class=HTMLResponse)
async def admin_login(request: Request):
    return templates.TemplateResponse("admin_login.html", {"request": request})

@app.get("/admin", response_class=HTMLResponse)
async def admin_dashboard(request: Request):
    try:
        admin = await get_current_admin(request)
        settings = await get_chatbot_settings(admin)
        return templates.TemplateResponse(
            "admin_dashboard.html",
            {"request": request, "admin": admin, "settings": settings}
        )
    except HTTPException:
        return RedirectResponse(url="/admin/login", status_code=303)

# Admin authentication dependency
async def get_current_admin(request: Request):
    token = request.cookies.get("admin_token") or request.headers.get("Authorization", "").replace("Bearer ", "")
    if not token:
        raise HTTPException(status_code=401, detail="Not authenticated")
    try:
        payload = jwt.decode(token, SECRET_KEY, algorithms=[ALGORITHM])
        email = payload.get("sub")
        if not email:
            raise HTTPException(status_code=401, detail="Invalid token")
        admin = get_admin_by_email(email)
        if not admin:
            raise HTTPException(status_code=401, detail="Admin not found")
        return admin
    except JWTError:
        raise HTTPException(status_code=401, detail="Invalid token")

# Settings management endpoints
@app.get("/api/settings", response_model=ChatbotSettings)
async def get_chatbot_settings(admin: dict = Depends(get_current_admin)):
    settings_data = get_settings()
    if not settings_data:
        raise HTTPException(status_code=404, detail="Settings not found")
    return ChatbotSettings(**settings_data)

@app.put("/api/settings")
async def update_chatbot_settings(settings: ChatbotSettings, admin: dict = Depends(get_current_admin)):
    if update_settings(settings.dict()):
        return {"message": "Settings updated successfully"}
    raise HTTPException(status_code=500, detail="Error updating settings")

@app.get("/")
async def root():
    """Root endpoint"""
    return {
        "message": "DLC Chatbot API is running",
        "version": "1.0.0",
        "endpoints": {
            "/ask": "POST - Ask a question",
            "/intents": "GET - List available intents",
            "/health": "GET - Check API health"
        }
    }

@app.post("/ask", response_model=Answer)
async def ask_question(question: Question):
    """Ask a question to the chatbot"""
    if not question.query:
        raise HTTPException(status_code=400, detail="Query cannot be empty")

    if not model:
        raise HTTPException(status_code=503, detail="Model not loaded")

    if not qa_data:
        raise HTTPException(status_code=503, detail="No Q&A data available")

    match = find_best_match(question.query, qa_data)
    if match:
        return Answer(
            answer=match['answer'],
            intent=match['intent'],
            confidence=match['confidence']
        )

    return Answer(
        answer="I'm sorry, I couldn't find a good match for your question. Could you please rephrase it?",
        confidence=0.0
    )

@app.get("/intents")
async def list_intents():
    """List all available intents"""
    intents = list(set(qa['intent'] for qa in qa_data))
    return {
        "total_intents": len(intents),
        "intents": intents
    }

@app.get("/health")
async def health_check():
    """Health check endpoint"""
    return {
        "status": "healthy",
        "model_loaded": model is not None,
        "qa_pairs_loaded": len(qa_data),
        "intents_available": len(set(qa['intent'] for qa in qa_data))
    }

# Run with: python -m uvicorn api:app --reload
if __name__ == "__main__":
    import uvicorn
    uvicorn.run(app, host="0.0.0.0", port=8000) 