import os
import glob
from typing import Optional, List
import pypdf
import time
import shutil
import ollama
from groq import Groq
from models import ChatbotSettings, ModelProvider
from langchain_community.vectorstores import FAISS
from langchain_community.embeddings import HuggingFaceEmbeddings
from langchain.text_splitter import RecursiveCharacterTextSplitter
from langchain_core.documents import Document
from dotenv import load_dotenv

# Load environment variables
load_dotenv()

class ChatService:
    def __init__(self):
        self.settings = None
        self.vectorstore = None
        self.embeddings = HuggingFaceEmbeddings(
            model_name="sentence-transformers/all-MiniLM-L6-v2",
            model_kwargs={'device': 'cpu'}
        )
        self.groq_client = None
        # Initialize vectorstore on startup (create only if it doesn't exist)
        self._ensure_vectorstore_exists()

    def rebuild_vectorstore(self, force: bool = False) -> bool:
        """Rebuild the vectorstore from documents
        
        Args:
            force: Force rebuild even if vectorstore exists
            
        Returns:
            bool: True if rebuild was performed, False otherwise
        """
        # If force is False and vectorstore exists, don't rebuild
        if not force and os.path.exists("vectorstore/db"):
            print("Vectorstore already exists. Set force=True to rebuild.")
            return False
            
        # Backup existing vectorstore if it exists
        if os.path.exists("vectorstore/db"):
            print("Backing up existing vectorstore...")
            backup_dir = f"vectorstore_backup_{int(time.time())}"
            os.makedirs(backup_dir, exist_ok=True)
            try:
                shutil.copytree("vectorstore/db", f"{backup_dir}/db")
                print(f"Backup created at {backup_dir}")
            except Exception as e:
                print(f"Warning: Failed to backup vectorstore: {str(e)}")
        
        # Create new vectorstore
        print("Creating new vectorstore...")
        try:
            # Create documents directory if it doesn't exist
            os.makedirs("documents", exist_ok=True)
            
            # Load documents
            documents = []
            for file in glob.glob("documents/*.pdf"):
                try:
                    pdf_reader = pypdf.PdfReader(file)
                    for i, page in enumerate(pdf_reader.pages):
                        text = page.extract_text() or ""
                        if text.strip():
                            documents.append(Document(
                                page_content=text,
                                metadata={"source": file, "page": i + 1}
                            ))
                    print(f"✅ Loaded {len(pdf_reader.pages)} pages from {file}")
                except Exception as e:
                    print(f"❌ Error loading {file}: {str(e)}")
            
            # If no documents found, create a sample document
            if not documents:
                print("No documents found. Creating sample document...")
                documents = [Document(
                    page_content="Welcome to DLC (Deep Learning Company). This is a sample document.",
                    metadata={"source": "sample"}
                )]
            
            # Create chunks
            text_splitter = RecursiveCharacterTextSplitter(
                chunk_size=500,
                chunk_overlap=50
            )
            texts = text_splitter.split_documents(documents)
            
            # Create and save vectorstore
            print(f"Creating vectorstore with {len(texts)} chunks...")
            os.makedirs("vectorstore", exist_ok=True)
            
            # If old vectorstore exists, delete it
            if os.path.exists("vectorstore/db"):
                print("Removing old vectorstore...")
                try:
                    shutil.rmtree("vectorstore/db")
                except Exception as e:
                    print(f"Warning: Failed to remove old vectorstore: {str(e)}")
            
            vectorstore = FAISS.from_documents(texts, self.embeddings)
            vectorstore.save_local("vectorstore/db")
            print("✅ Vectorstore created successfully!")
            
            # Reset instance vectorstore to use the new one
            self.vectorstore = vectorstore
            return True
            
        except Exception as e:
            print(f"❌ Error creating vectorstore: {str(e)}")
            raise

    def _ensure_vectorstore_exists(self):
        """Ensure the vectorstore exists, create it if it doesn't"""
        if not os.path.exists("vectorstore/db"):
            print("Vectorstore not found. Creating initial vectorstore...")
            self.rebuild_vectorstore(force=True)
        else:
            print("Using existing vectorstore. To rebuild, call rebuild_vectorstore(force=True)")

    def initialize(self, settings: ChatbotSettings):
        """Initialize the chat service with given settings"""
        self.settings = settings
        if settings.provider == ModelProvider.GROQ:
            api_key = settings.api_key or os.getenv("GROQ_API_KEY")
            if not api_key:
                raise ValueError("Groq API key is required but not provided")
            self.groq_client = Groq(api_key=api_key)

    def get_relevant_context(self, query: str) -> str:
        """Get relevant context from the vectorstore"""
        if not self.vectorstore:
            try:
                self.vectorstore = FAISS.load_local(
                    "vectorstore/db", 
                    self.embeddings,
                    allow_dangerous_deserialization=True  # We trust our own vectorstore
                )
            except Exception as e:
                print(f"Error loading vectorstore: {str(e)}")
                return ""
        
        try:
            docs = self.vectorstore.similarity_search(query, k=3)
            return "\n\n".join([doc.page_content for doc in docs])
        except Exception as e:
            print(f"Error during similarity search: {str(e)}")
            return ""

    async def generate_response(self, query: str, context: str) -> str:
        """Generate response using the configured provider"""
        if not self.settings:
            raise ValueError("Chat service not initialized with settings")

        # Handle greetings and identity questions
        greeting_phrases = ["hello", "hi", "hey", "greetings", "good morning", "good afternoon", "good evening", "how are you"]
        identity_keywords = ["who are you", "what are you", "your name", "who is this", "who am i talking to", "who is chatting", "who created you"]
        
        if any(phrase in query.lower() for phrase in greeting_phrases):
            return "Hello! I'm DLC's AI assistant. How can I help you today?"
        
        if any(keyword in query.lower() for keyword in identity_keywords):
            return "I am DLC's AI assistant, designed to help with questions about Distance LEarning Center and provide general information. How can I assist you today?"

        prompt = f"""You are a helpful AI assistant for DLC (Distance Learning). Use the following context to answer the question. 
If you cannot answer the question based on the context, say so.

Context:
{context}

Question: {query}

Answer:"""

        try:
            if self.settings.provider == ModelProvider.OLLAMA:
                response = ollama.chat(
                    model=self.settings.model_name,
                    messages=[
                        {"role": "system", "content": "You are a helpful AI assistant for DLC."},
                        {"role": "user", "content": prompt}
                    ],
                    options={
                        "temperature": self.settings.temperature,
                        "top_p": self.settings.top_p,
                        "num_predict": self.settings.max_tokens
                    }
                )
                return response['message']['content']

            elif self.settings.provider == ModelProvider.GROQ:
                if not self.groq_client:
                    raise ValueError("Groq client not initialized. API key may be missing.")
                
                chat_completion = self.groq_client.chat.completions.create(
                    messages=[
                        {"role": "system", "content": "You are a helpful AI assistant for DLC."},
                        {"role": "user", "content": prompt}
                    ],
                    model=self.settings.model_name,
                    temperature=self.settings.temperature,
                    max_tokens=self.settings.max_tokens,
                    top_p=self.settings.top_p,
                )
                return chat_completion.choices[0].message.content

            raise ValueError(f"Unsupported provider: {self.settings.provider}")
        
        except Exception as e:
            error_msg = str(e).lower()
            if "connection refused" in error_msg or "failed to connect" in error_msg:
                return "I'm sorry, but the language model service is currently unavailable. Please try again later."
            elif "model not found" in error_msg:
                return "I'm sorry, but the required language model is not available. Please check the model configuration."
            else:
                return f"I'm sorry, but I encountered an error while processing your request: {str(e)}"

    async def get_response(self, query: str) -> str:
        """Convenience method to get a response in one call"""
        context = self.get_relevant_context(query)
        return await self.generate_response(query, context)

# Create a global instance
chat_service = ChatService() 