from pymongo import MongoClient
from datetime import datetime
import os
from dotenv import load_dotenv

# Load environment variables
load_dotenv()

# MongoDB connection settings
MONGO_URI = os.getenv("MONGO_URI")
DB_NAME = "dlc_chatbot"

# Initialize MongoDB client
client = MongoClient(MONGO_URI)
db = client[DB_NAME]

# Collections
conversations_collection = db.conversations
chat_sessions_collection = db.chat_sessions
settings_collection = db.settings
admin_collection = db.admins

# Create indexes
try:
    # Create indexes for better query performance
    conversations_collection.create_index([("session_id", 1)])
    conversations_collection.create_index([("timestamp", -1)])
    chat_sessions_collection.create_index([("created_at", -1)])
    admin_collection.create_index([("email", 1)], unique=True)
    print("Database indexes created successfully")
except Exception as e:
    print(f"Error creating indexes: {str(e)}")

def insert_conversation(session_id: str, question: str, answer: str, source: str = "intention"):
    """Insert a conversation entry"""
    try:
        print(f"\nInserting conversation:")
        print(f"Session ID: {session_id}")
        print(f"Question: {question}")
        print(f"Answer: {answer}")
        
        # Create or update session
        chat_sessions_collection.update_one(
            {"session_id": session_id},
            {
                "$setOnInsert": {
                    "created_at": datetime.utcnow(),
                    "session_id": session_id
                }
            },
            upsert=True
        )
        
        # Insert conversation
        result = conversations_collection.insert_one({
            "session_id": session_id,
            "question": question,
            "answer": answer,
            "timestamp": datetime.utcnow(),
            "source": source
        })
        
        print(f"Conversation inserted successfully. ID: {result.inserted_id}")
        return True
    except Exception as e:
        print(f"Error inserting conversation: {str(e)}")
        return False

def get_session_list():
    """Get list of all chat sessions"""
    try:
        return list(chat_sessions_collection.find({}, {
            "_id": 0,
            "session_id": 1,
            "created_at": 1
        }).sort("created_at", -1))
    except Exception as e:
        print(f"Error getting session list: {str(e)}")
        return []

def get_session_history(session_id: str):
    """Get conversation history for a session"""
    try:
        return list(conversations_collection.find(
            {"session_id": session_id},
            {"_id": 0}
        ).sort("timestamp", 1))
    except Exception as e:
        print(f"Error getting session history: {str(e)}")
        return []

def get_settings():
    """Get chatbot settings"""
    try:
        return settings_collection.find_one({}, {"_id": 0})
    except Exception as e:
        print(f"Error getting settings: {str(e)}")
        return None

def update_settings(new_settings: dict):
    """Update chatbot settings"""
    try:
        settings_collection.replace_one({}, new_settings, upsert=True)
        return True
    except Exception as e:
        print(f"Error updating settings: {str(e)}")
        return False 