from pydantic import BaseModel, Field
from typing import List, Optional, Dict
from enum import Enum

class ModelProvider(str, Enum):
    SENTENCE_TRANSFORMERS = "sentence-transformers"
    INTENTION = "intention"

class ChatbotSettings(BaseModel):
    # Model Settings
    model_name: str = Field(
        default="paraphrase-MiniLM-L6-v2",
        description="The name of the sentence-transformer model to use"
    )
    similarity_threshold: float = Field(
        default=0.6,
        ge=0.0,
        le=1.0,
        description="Minimum similarity score required for a match (0.0 to 1.0)"
    )
    
    # Response Settings
    default_response: str = Field(
        default="I'm sorry, I couldn't find a good match for your question. Could you please rephrase it?",
        description="Default response when no good match is found"
    )
    greeting_message: str = Field(
        default="Hello! I'm the DLC chatbot. How can I help you today?",
        description="Initial greeting message shown to users"
    )
    
    # Q&A Data Settings
    enabled_qa_files: List[str] = Field(
        default=[
            "general_chat.json",
            "dlc_qa.json",
            "program_qa.json",
            "nursing_qa.json"
        ],
        description="List of Q&A JSON files to load"
    )
    
    # Cache Settings
    enable_response_cache: bool = Field(
        default=True,
        description="Whether to cache responses for frequently asked questions"
    )
    cache_max_size: int = Field(
        default=1000,
        description="Maximum number of responses to keep in cache"
    )
    
    # Logging Settings
    debug_mode: bool = Field(
        default=True,
        description="Enable detailed debug logging"
    )
    log_conversations: bool = Field(
        default=True,
        description="Whether to log conversations to the database"
    )
    
    class Config:
        title = "DLC Chatbot Settings"
        description = "Configuration settings for the DLC intention-based chatbot" 