import os
import glob
import json
from typing import List, Dict, Any

def load_qa_data(directory: str) -> List[Dict[str, Any]]:
    """Load and flatten all Q&A pairs from JSON files in the directory."""
    qa_data = []
    json_files = glob.glob(os.path.join(directory, "*.json"))
    for json_path in json_files:
        try:
            with open(json_path, 'r', encoding='utf-8') as f:
                data = json.load(f)
                # Handle various JSON structures
                if isinstance(data, dict):
                    for key, items in data.items():
                        if isinstance(items, list):
                            for item in items:
                                if isinstance(item, dict) and 'intent' in item and 'questions' in item:
                                    answers = item.get('answers', [item.get('answer', '')])
                                    if not isinstance(answers, list):
                                        answers = [answers]
                                    for q in item['questions']:
                                        qa_data.append({
                                            'intent': item['intent'],
                                            'question': q,
                                            'answers': answers
                                        })
                elif isinstance(data, list):
                    for item in data:
                        if isinstance(item, dict) and 'intent' in item and 'questions' in item:
                            answers = item.get('answers', [item.get('answer', '')])
                            if not isinstance(answers, list):
                                answers = [answers]
                            for q in item['questions']:
                                qa_data.append({
                                    'intent': item['intent'],
                                    'question': q,
                                    'answers': answers
                                })
        except Exception as e:
            print(f"Error processing {json_path}: {str(e)}")
    return qa_data

def find_best_match(query: str, qa_data: List[Dict[str, Any]]) -> Dict[str, Any]:
    """Find the best matching Q&A pair for a given query using simple string similarity."""
    query_lower = query.strip().lower()
    # Exact match first
    for qa in qa_data:
        if query_lower == qa['question'].strip().lower():
            return qa
    # Substring match
    for qa in qa_data:
        if query_lower in qa['question'].strip().lower() or qa['question'].strip().lower() in query_lower:
            return qa
    # Fallback: naive word overlap
    query_words = set(query_lower.split())
    best_qa = None
    best_overlap = 0
    for qa in qa_data:
        qa_words = set(qa['question'].strip().lower().split())
        overlap = len(query_words & qa_words)
        if overlap > best_overlap:
            best_overlap = overlap
            best_qa = qa
    if best_overlap > 0:
        return best_qa
    return None

def main():
    DOCUMENTS_DIR = "documents"
    qa_data = load_qa_data(DOCUMENTS_DIR)
    print(f"\nIntent-based Chatbot initialized! Type 'quit' to exit.")
    while True:
        query = input("\nYour question: ").strip()
        if query.lower() == 'quit':
            break
        if not query:
            continue
        match = find_best_match(query, qa_data)
        if match:
            # If multiple answers, pick the first (or random if you want variety)
            answer = match['answers'][0] if match['answers'] else "No answer available."
            print(f"\nAnswer: {answer}")
        else:
            print("\nI'm sorry, I couldn't find a good match for your question. Could you please rephrase it?")

if __name__ == "__main__":
    main()
